<?php

namespace App\Services\Billar\PaymentHistory;

use App\Helpers\App\Traits\SmsHelper;
use App\Jobs\InvoiceAttachmentJob;
use App\Models\App\PaymentMethods\PaymentMethod;
use App\Models\Billar\Invoice\Invoice;
use App\Models\Billar\PaymentHistory\PaymentHistory;
use App\Models\Core\Status;
use App\Services\App\SmsSetting\TwilioService;
use App\Services\Billar\ApplicationBaseService;
use Stripe\{Charge, Customer, Stripe};
use Illuminate\Support\Facades\Storage;
use PDF;
use PHPUnit\Util\Xml\Exception;
use Razorpay\Api\Api;

class InvoicePaymentService extends ApplicationBaseService
{
    use SmsHelper;

    public function __construct(Invoice $invoice)
    {
        $this->model = $invoice;
    }

    public function stripePayment(): self
    {
        Stripe::setApiKey(config()->get('services.stripe.secret_key'));
        $customer = Customer::create([
            'email' => request('stripeEmail'),
            'source' => request('stripeToken')
        ]);

        $charge = Charge::create([
            'customer' => $customer->id,
            'amount' => request('amount') * 100,
            'currency' => 'usd',
        ]);

        //succeeded
        if ($charge->status == 'succeeded') {
            $this->updateInvoice()
                ->paymentHistory();
        }
        return $this;

    }

    public function paypalPayment(): self
    {
        return $this->updateInvoice()
            ->paymentHistory();
    }

    public function paystackPayment(): InvoicePaymentService
    {
        $this->updateInvoice()
            ->paymentHistory();
        return $this;
    }

    public function razorpayPayment(): InvoicePaymentService
    {
        $api = new Api(config()->get('services.razorpay.razorpay_key'), config()->get('services.razorpay.razorpay_secret'));
        $payment = $api->payment->fetch(request()->get('razorpay_payment_id'));
        if (request()->get('razorpay_payment_id')) {
            $capture = $api->payment->fetch(request()->get('razorpay_payment_id'))->capture(array('amount' => $payment['amount']));
            if ($capture['status'] == 'captured') {
                $this->updateInvoice()
                    ->paymentHistory();
            }
        }

        return $this;


    }

    public function updateInvoice(): self
    {
        $invoice = $this->model::query()
            ->where('id', request('invoice_id'));
        $invoice->decrement('due_amount', request('amount'));
        $invoice->increment('received_amount', request('amount'));
        $invoice->update(['status_id' => Status::findByNameAndType('status_paid', 'invoice')->id]);
        return $this;
    }

    public function paymentHistory(): self
    {
        $payment = PaymentHistory::create([
            'invoice_id' => request('invoice_id'),
            'payment_method_id' => PaymentMethod::where('alias', request('payment_type'))->first()->id,
            'received_on' => date('Y-m-d'),
            'amount' => request('amount'),
            'note' => request('reference') ? 'Reference : '.request('reference') : null,
        ]);

        $invoiceInfo = $payment->load('invoice')->invoice;

        $this->setAttribute('file_path', 'public/pdf/invoice_' . $invoiceInfo->id . '.pdf')
            ->pdfGenerate($invoiceInfo);
        InvoiceAttachmentJob::dispatch($invoiceInfo, 'payment_recived')->onQueue('high');

        //send sms to client after payment received
        $settings = resolve(TwilioService::class)->getData();

        if (isset($invoiceInfo->client->profile->contact) && isset($settings['send_auto_sms'])) {
            $message = __t('dear') . $invoiceInfo->client->full_name . ','.__t('your_payment_of') . request('amount') . __t('has_been_received_thank_you');
            $this->sendSms($invoiceInfo->client->profile->contact, $message);
        }
        return $this;
    }

    public function pdfGenerate($invoiceInfo): self
    {
        $pdf = PDF::loadView('invoices.invoice-generate', [
            'invoice' => $invoiceInfo
        ]);

        $output = $pdf->output();
        $filePath = $this->getAttribute('file_path');
        Storage::put($filePath, $output);
        return $this;
    }

}